package com.joshvm.ws;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.Random;
import java.util.Vector;

import javax.microedition.io.Connector;
import javax.microedition.io.SocketConnection;

import com.joshvm.utils.Logger;
import com.joshvm.utils.Utils;
import org.json.me.JSONArray;
import org.json.me.JSONException;
import org.json.me.JSONObject;

import com.joshvm.ConnectionParameters;

/**
 * CLDC环境下的WebSocket客户端实现 支持长帧、帧组装、异步处理等完整功能
 */
public class WebSocketClient {

	// WebSocket操作码
	private static final byte OPCODE_CONTINUATION = 0x0;
	private static final byte OPCODE_TEXT = 0x1;
	private static final byte OPCODE_BINARY = 0x2;
	private static final byte OPCODE_CLOSE = 0x8;
	private static final byte OPCODE_PING = 0x9;
	private static final byte OPCODE_PONG = 0xA;

	// WebSocket状态
	private static final int STATE_CONNECTING = 0;
	private static final int STATE_OPEN = 1;
	private static final int STATE_CLOSING = 2;
	private static final int STATE_CLOSED = 3;

	// 配置参数
	private static final int POLL_TIMEOUT = 100; // 轮询超时时间(ms)
	private static final int MAX_FRAME_SIZE = 16384; // 16KB（原64KB）
	private static final int BUFFER_SIZE = 2048; // 2KB（原8KB）

	private SocketConnection socket;
	private InputStream inputStream;
	private OutputStream outputStream;
	private int state = STATE_CLOSED;
	private String gcfProtocol;
	private String url;
	private String host;
	private int port;
	private String path;
	private WebSocketListener listener;
	private Random random;
	private Thread readerThread;
	private Thread writerThread;
	private boolean running = false;
	private boolean usePolling = false;

	private FragmentedMessage activeFragment = null;
	private boolean interrupted = false; // 自定义中断标志

	// 消息发送队列
	private final Vector sendQueue = new Vector();
	private final Object sendLock = new Object();
//	private static final long FRAGMENT_TIMEOUT = 30000; // 30秒超时
//	private long lastFragmentTime;

	// 帧状态跟踪
	private WebSocketFrame incompleteFrame = null;
	private int expectedPayloadLength = 0;
	private int receivedPayloadLength = 0;
	private byte[] framePayloadBuffer = null;
	
	private ConnectionParameters param;

	// 帧组装相关
	private final Object frameLock = new Object();

	// 读取缓冲区
	private final CircularBuffer readBuffer = new CircularBuffer(BUFFER_SIZE * 2);

	/**
	 * 构造函数
	 * 
	 * @param url      WebSocket服务器URL
	 * @param listener 事件监听器
	 */
	public WebSocketClient(ConnectionParameters param, WebSocketListener listener) {
		this(param.getUrl(), listener, false);
		this.param = param;
	}

	/**
	 * 构造函数
	 * 
	 * @param url        WebSocket服务器URL
	 * @param listener   事件监听器
	 * @param usePolling 是否使用轮询模式（非阻塞）
	 */
	public WebSocketClient(String url, WebSocketListener listener, boolean usePolling) {
		this.url = url;
		this.listener = listener;
		this.usePolling = usePolling;
		this.random = new Random();
		parseUrl(url);
	}

	/**
	 * 解析WebSocket URL
	 */
	private void parseUrl(String url) {
		if (url.startsWith("ws://")) {
			url = url.substring(5);
			gcfProtocol = "socket://";
		} else if (url.startsWith("wss://")) {
			url = url.substring(6);
			gcfProtocol = "ssl://";
		} else {
			throw new IllegalArgumentException("unsupport protocol");
		}

		int slashIndex = url.indexOf('/');
		String hostPort;
		if (slashIndex != -1) {
			hostPort = url.substring(0, slashIndex);
			this.path = url.substring(slashIndex);
		} else {
			hostPort = url;
			this.path = "/";
		}

		int colonIndex = hostPort.indexOf(':');
		if (colonIndex != -1) {
			this.host = hostPort.substring(0, colonIndex);
			this.port = Integer.parseInt(hostPort.substring(colonIndex + 1));
		} else {
			this.host = hostPort;
			if (gcfProtocol.equals("ssl://"))
				this.port = 443;
			else
				this.port = 80;
		}
	}

	/**
	 * 连接到WebSocket服务器
	 */
	public synchronized boolean connect() throws IOException {
		if (state != STATE_CLOSED) {
			throw new IllegalStateException("WebSocket is already connected or connecting");
		}

//        initCA();

		state = STATE_CONNECTING;

		try {
			if (Logger.isDebug()) Logger.debug("ws url="+gcfProtocol+host+":"+port);
			// 建立TCP连接
			socket = (SocketConnection) Connector.open(gcfProtocol + host + ":" + port);
			inputStream = socket.openInputStream();
			outputStream = socket.openOutputStream();

			// 执行WebSocket握手
			if (!performHandshake()) {
				return false;
			}

			state = STATE_OPEN;
			running = true;

			// 启动读取线程
			readerThread = new Thread(new Runnable() {
				public void run() {
					if (usePolling) {
						pollingReadLoop();
					} else {
						blockingReadLoop();
					}
				}
			}, "WebSocket-Reader");
			readerThread.start();

			// 启动写入线程
			writerThread = new Thread(new Runnable() {
				public void run() {
					writeLoop();
				}
			}, "WebSocket-Writer");
			writerThread.start();

			if (listener != null) {
				listener.onOpen();
			}
			return true;

		} catch (Exception e) {
			if (Logger.isInfo()) Logger.info("ws conn "+host+" failed for "+e);
			e.printStackTrace();
			state = STATE_CLOSED;
			cleanup();
		}
		return false;
	}

//	/**
//	 * 添加ca证书验证
//	 */
//	private void initCA() {
//		try {
////			String cert = Utils.readCertFromResource(Constant.CRT);
////			String key = Utils.readPrivateKeyFromResource(Constant.KEY);
////			KeyStore.initWebPublicKeystoreLocation(this.getClass(), Constant.KS);
////			PrivateKeyStore ks = KeyStore.selectPrivateKeyStore(null);
////			ks.load(((UserKeyStoreParam) UserKeyStoreParam.Build()).addCertificate(cert).setPrivateKey(key));
//			if (Logger.isDebug()) Logger.debug("==========init ca success==========");
//		} catch (Exception e) {
//			if (Logger.isDebug()) Logger.debug("==========init ca  failed==========");
//			e.printStackTrace();
//		}
//	}

	private void fillRandomBytes(byte[] bytes) {
		for (int i = 0; i < bytes.length; i++) {
			bytes[i] = (byte) random.nextInt(256);
		}
	}

	/**
	 * 执行WebSocket握手
	 */
	private boolean performHandshake() throws IOException {

		byte[] keyBytes = new byte[16];
		fillRandomBytes(keyBytes);
		String key = Base64.encode(keyBytes);

		StringBuffer request = new StringBuffer();
		request.append("GET ").append(path).append(" HTTP/1.1\r\n");
		request.append("Host: ").append(host).append(":").append(port).append("\r\n");
		request.append("Upgrade: websocket\r\n");
		request.append("Connection: Upgrade\r\n");
		request.append("Sec-WebSocket-Key: ").append(key).append("\r\n");
		request.append("Sec-WebSocket-Version: 13\r\n");
		
		String header = param.getHeader();
		if(header != null && header.length()>0) {
			try {
				JSONObject o = new JSONObject(header);
				JSONArray keys = o.names();
				for(int i =0 ;i<keys.length();i++) {
					String k = (String) keys.get(i);
					Object v = o.get(k);
					request.append(k).append(": ")  .append(String.valueOf(v)).append("\r\n");
				}
			} catch (JSONException e) {
				e.printStackTrace();
			}
		}
		request.append("\r\n");
		if (Logger.isDebug()) Logger.debug("ws client: shake request=" + request.toString());

		outputStream.write(request.toString().getBytes());
		outputStream.flush();

		String response = readHttpResponse();
		if (Logger.isDebug()) Logger.debug("ws client: shake response=" + response);

		if (!response.startsWith("HTTP/1.1 101")) {
			/* throw new IOException */
			if (Logger.isInfo()) Logger.info("ws client: handshake failed: " + response);
			return false;
		}
		String lowerResponse = response.toLowerCase();
		if (lowerResponse.indexOf("upgrade: websocket") == -1 || lowerResponse.indexOf("connection: upgrade") == -1) {
			if (Logger.isInfo()) Logger.info("ws client: invalid handshake response");
			return false;
		}

		return true;
	}

	/**
	 * 读取HTTP响应
	 */
	private String readHttpResponse() throws IOException {
		StringBuffer response = new StringBuffer();
		int b;
		boolean inHeader = true;
		int consecutiveNewlines = 0;

		while (inHeader && (b = inputStream.read()) != -1) {
			response.append((char) b);
			if (b == '\n') {
				consecutiveNewlines++;
				if (consecutiveNewlines == 2) {
					inHeader = false;
				}
			} else if (b != '\r') {
				consecutiveNewlines = 0;
			}
		}

		return response.toString();
	}

	/**
	 * 发送文本消息（支持长消息自动分片）
	 */
	public boolean sendText(String message) throws IOException {
		if (state != STATE_OPEN) {
//            throw new IllegalStateException("WebSocket is not open");
			return false;
		}
		if (Logger.isDebug()) Logger.debug("ws client: send msg=" + message);
		byte[] data = message.getBytes("UTF-8");
		if (data.length <= MAX_FRAME_SIZE) {
			queueFrame(new OutgoingFrame(OPCODE_TEXT, data, true));
		} else {
			// 分片发送长消息
			sendFragmentedMessage(OPCODE_TEXT, data);
		}
		return true;
	}

	/**
	 * 发送二进制消息（支持长消息自动分片）
	 */
	public boolean sendBinary(byte[] data) throws IOException {
		if (state != STATE_OPEN) {
//            throw new IllegalStateException("WebSocket is not open");
			return false;
		}

		if (data.length <= MAX_FRAME_SIZE) {
			queueFrame(new OutgoingFrame(OPCODE_BINARY, data, true));
		} else {
			sendFragmentedMessage(OPCODE_BINARY, data);
		}
		return true;
	}

	/**
	 * 发送分片消息
	 */
	private void sendFragmentedMessage(byte opcode, byte[] data) {
		int offset = 0;
		boolean first = true;

		while (offset < data.length) {
			int chunkSize = Math.min(MAX_FRAME_SIZE, data.length - offset);
			byte[] chunk = new byte[chunkSize];
			System.arraycopy(data, offset, chunk, 0, chunkSize);

			byte frameOpcode = first ? opcode : OPCODE_CONTINUATION;
			boolean fin = (offset + chunkSize >= data.length);

			queueFrame(new OutgoingFrame(frameOpcode, chunk, fin));

			offset += chunkSize;
			first = false;
		}
	}

	/**
	 * 发送Ping帧
	 */
	public void sendPing(byte[] data) throws IOException {
		if (state != STATE_OPEN) {
			throw new IllegalStateException("WebSocket is not open");
		}

		if (data == null)
			data = new byte[0];
		if (data.length > 125) {
			throw new IllegalArgumentException("Ping payload too large");
		}

		queueFrame(new OutgoingFrame(OPCODE_PING, data, true));
	}

	/**
	 * 发送Pong帧
	 */
	public void sendPong(byte[] data) {
		if (state == STATE_OPEN) {
			if (data == null)
				data = new byte[0];
			queueFrame(new OutgoingFrame(OPCODE_PONG, data, true));
		}
	}

	/**
	 * 将帧加入发送队列
	 */
	private void queueFrame(OutgoingFrame frame) {
		synchronized (sendLock) {
			sendQueue.addElement(frame);
			sendLock.notify();
		}
	}

	/**
	 * 轮询模式读取循环（非阻塞）
	 */
	private void pollingReadLoop() {
		byte[] buffer = new byte[BUFFER_SIZE];
		if (Logger.isDebug()) Logger.debug("ws client: pollingReadLoop start");

		try {
			while (running && state == STATE_OPEN && !interrupted) {
				try {
					int available = inputStream.available();
					if (Logger.isTrace()) Logger.trace("ws client: pollread input steam avaliable=" + available);

					if (available > 0) {
						int toRead = Math.min(available, buffer.length);
						int bytesRead = inputStream.read(buffer, 0, toRead);
						if (Logger.isDebug()) Logger.debug("ws client: pollread bytes "+bytesRead);
						if (bytesRead > 0) {
							readBuffer.write(buffer, 0, bytesRead);
							processBufferedData();
						}
					} else {
						// 没有数据可读，短暂休眠
						try {
							Thread.sleep(POLL_TIMEOUT);
						} catch (InterruptedException e) {
							break;
						}
					}
				} catch (IOException e) {
					if (Logger.isDebug()) Logger.debug("ws client: pollread end for io "+e);
					if (running) {
						notifyError(e);
					}
					break;
				} catch (Exception ee) {
					// InterruptedException?
					if (Logger.isDebug()) Logger.debug("ws client: pollread end for "+ee);
				}
			}
		} finally {
			if (listener != null) {
				listener.onClose(1000, "");
			}
			close();
		}
	}

	/**
	 * 阻塞模式读取循环
	 */
	private void blockingReadLoop() {
		byte[] buffer = new byte[BUFFER_SIZE];
		if (Logger.isDebug()) Logger.debug("ws client: blockingReadLoop start");

		try {
			while (running && state == STATE_OPEN && !interrupted) {
				try {
					int bytesRead = inputStream.read(buffer);
					if (Logger.isDebug()) Logger.debug("ws client: blockread bytes "+bytesRead);
					if (bytesRead == -1) {
						break; // 连接关闭
					}
					if (bytesRead > 0) {
						//if (Logger.isDebug()) Logger.debug("Read " + bytesRead + " bytes from wssocket");
						readBuffer.write(buffer, 0, bytesRead);

						// 立即尝试处理数据
						processBufferedData();
					}
				} catch (IOException e) {
					if (Logger.isDebug()) Logger.debug("ws client: blockread end for io "+e);
					if (running) {
						notifyError(e);
					}
					break;
				} catch (Exception ee) {
					// InterruptedException?
					if (Logger.isDebug()) Logger.debug("ws client: blockread end for "+ee);
				}
			}
		} finally {
			if (listener != null) {
				listener.onClose(1000, "");
			}
			close();
		}
	}

	/**
	 * 尝试从缓冲区读取一个完整的帧
	 */
	private WebSocketFrame tryReadFrame() throws IOException {
		// 如果有未完成的帧，继续读取其payload
		if (incompleteFrame != null) {
			return continueReadingIncompleteFrame();
		}

		// 至少需要2字节的基本头部
		if (readBuffer.available() < 2) {
			return null;
		}

		int startPos = readBuffer.getReadPosition();

		try {
			// 读取第一个字节
			int firstByte = readBuffer.read();
			boolean fin = (firstByte & 0x80) != 0;
			byte opcode = (byte) (firstByte & 0x0F);

			// 读取第二个字节
			int secondByte = readBuffer.read();
			boolean masked = (secondByte & 0x80) != 0;
			int payloadLength = secondByte & 0x7F;
			//int unextendLength = payloadLength;

			// 处理扩展长度
			if (payloadLength == 126) {
				if (readBuffer.available() < 2) {
					readBuffer.setReadPosition(startPos);
					return null;
				}
				int b1 = readBuffer.read() & 0xFF;
				int b2 = readBuffer.read() & 0xFF;
				payloadLength = (b1 << 8) | b2;
				//if (Logger.isDebug()) Logger.debug("Extended length: " + payloadLength);
			} else if (payloadLength == 127) {
				if (readBuffer.available() < 8) {
					readBuffer.setReadPosition(startPos);
					return null;
				}
				// 读取8字节长度，但只使用低32位
				readBuffer.read();
				readBuffer.read();
				readBuffer.read();
				readBuffer.read();
				int b1 = readBuffer.read() & 0xFF;
				int b2 = readBuffer.read() & 0xFF;
				int b3 = readBuffer.read() & 0xFF;
				int b4 = readBuffer.read() & 0xFF;
				payloadLength = (b1 << 24) | (b2 << 16) | (b3 << 8) | b4;
				//if (Logger.isDebug()) Logger.debug("Extended length (64-bit): " + payloadLength);
			}

			if (Logger.isDebug()) Logger.debug("ws client: read frame header: fin=" + fin + " opcode=" + opcode + " masked=" + masked + " length=" + payloadLength);

			// 验证长度
			if (payloadLength < 0 || payloadLength > FragmentedMessage.MAX_MESSAGE_SIZE) {
				throw new IOException("ws client: read frame invalid payload length=" + payloadLength);
			}

			// 处理掩码
			byte[] maskKey = null;
			if (masked) {
				if (readBuffer.available() < 4) {
					readBuffer.setReadPosition(startPos);
					return null;
				}
				maskKey = new byte[4];
				for (int i = 0; i < 4; i++) {
					maskKey[i] = (byte) readBuffer.read();
				}
			}

			// 检查payload数据是否完整
			int availablePayload = readBuffer.available();
			if (availablePayload < payloadLength) {
				if (Logger.isDebug()) Logger.debug("ws client: read frame incomplete frame (" + availablePayload + "/" + payloadLength + ")");

				// 创建未完成的帧状态
				incompleteFrame = new WebSocketFrame(fin, opcode, null);
				expectedPayloadLength = payloadLength;
				receivedPayloadLength = 0;
				framePayloadBuffer = new byte[payloadLength];

				// 读取当前可用的payload数据
				for (int i = 0; i < availablePayload; i++) {
					framePayloadBuffer[receivedPayloadLength++] = (byte) readBuffer.read();
				}

				// 应用掩码到已读取的数据
				if (masked && maskKey != null) {
					for (int i = 0; i < receivedPayloadLength; i++) {
						framePayloadBuffer[i] ^= maskKey[i % 4];
					}
				}

				// 保存掩码键以便后续使用
				incompleteFrame.maskKey = maskKey;

				return null; // 等待更多数据
			}

			// 完整帧 - 一次性读取所有payload
			byte[] payload = new byte[payloadLength];
			for (int i = 0; i < payloadLength; i++) {
				payload[i] = (byte) readBuffer.read();
			}

			// 解掩码
			if (masked && maskKey != null) {
				for (int i = 0; i < payloadLength; i++) {
					payload[i] ^= maskKey[i % 4];
				}
			}

			if (Logger.isDebug()) Logger.debug("ws client: read parse complete frame FIN=" + fin + ", Opcode=" + opcode + ", Length=" + payloadLength);

			return new WebSocketFrame(fin, opcode, payload);

		} catch (Exception e) {
			readBuffer.setReadPosition(startPos);
			if (Logger.isInfo()) Logger.info("ws client: read frame parsing error: " + e.getMessage());
			e.printStackTrace();
			throw new IOException("ws client: read frame parsing failed: " + e.getMessage());
		}
	}

	/**
	 * 继续读取未完成帧的payload数据
	 */
	private WebSocketFrame continueReadingIncompleteFrame() throws IOException {
		int remainingBytes = expectedPayloadLength - receivedPayloadLength;
		int availableBytes = readBuffer.available();

		if (availableBytes == 0) {
			return null; // 等待更多数据
		}

		int bytesToRead = Math.min(remainingBytes, availableBytes);

		if (Logger.isDebug()) Logger.debug("ws client: read continuing incomplete frame: need " + remainingBytes + ", available " + availableBytes + ", reading " + bytesToRead);

		// 读取更多payload数据
		for (int i = 0; i < bytesToRead; i++) {
			framePayloadBuffer[receivedPayloadLength++] = (byte) readBuffer.read();
		}

		// 应用掩码到新读取的数据
		if (incompleteFrame.maskKey != null) {
			for (int i = receivedPayloadLength - bytesToRead; i < receivedPayloadLength; i++) {
				framePayloadBuffer[i] ^= incompleteFrame.maskKey[i % 4];
			}
		}

		// 检查是否完成
		if (receivedPayloadLength >= expectedPayloadLength) {
			if (Logger.isDebug()) Logger.debug("ws client: read incomplete frame completed: total length " + receivedPayloadLength);

			// 创建完整的帧
			WebSocketFrame completeFrame = new WebSocketFrame(incompleteFrame.fin, incompleteFrame.opcode,
					framePayloadBuffer);

			// 重置状态
			incompleteFrame = null;
			expectedPayloadLength = 0;
			receivedPayloadLength = 0;
			framePayloadBuffer = null;

			return completeFrame;
		}

		return null; // 仍需更多数据
	}

	/**
	 * 处理接收到的帧
	 */
	private void handleFrame(WebSocketFrame frame) {
		if (Logger.isTrace()) Logger.trace("ws client: handle frame fin=" + frame.fin + " opcode=" + frame.opcode + " length=" + frame.payload.length);

		try {
			switch (frame.opcode) {
			case OPCODE_TEXT:
			case OPCODE_BINARY:
				handleDataFrame(frame);
				break;

			case OPCODE_CONTINUATION:
				handleContinuationFrame(frame);
				break;

			case OPCODE_PING:
				if (Logger.isDebug()) Logger.debug("ws client: handle frame recv PING");
				sendPong(frame.payload);
				if (listener != null) {
					listener.onPing(frame.payload);
				}
				break;

			case OPCODE_PONG:
				if (Logger.isDebug()) Logger.debug("ws client: handle frame recv PONG");
				if (listener != null) {
					listener.onPong(frame.payload);
				}
				break;

			case OPCODE_CLOSE:
				if (Logger.isDebug()) Logger.debug("ws client: handle frame recv CLOSE");
				handleCloseFrame(frame);
				break;

			default:
				if (Logger.isDebug()) Logger.debug("ws client: handle frame recv unknown opcode " + frame.opcode);
				break;
			}
		} catch (Exception e) {
			notifyError(e);
		}
	}

	/**
	 * 处理数据帧（文本或二进制）
	 */
	private void handleDataFrame(WebSocketFrame frame) throws IOException {
		synchronized (frameLock) {
			if (frame.fin) {
				// 完整消息
				if (activeFragment != null) {
					// 这不应该发生，但为了健壮性处理
					if (Logger.isInfo()) Logger.info("ws client: handle data frame complete frame while fragment active!!!");
					activeFragment.clear();
					activeFragment = null;
				}
				processCompleteMessage(frame.opcode, frame.payload);
			} else {
				// 分片消息的开始
				if (activeFragment != null) {
					if (Logger.isDebug()) Logger.debug("ws client: handle data frame start new fragment while previous active");
					activeFragment.clear();
				}
				activeFragment = new FragmentedMessage(frame.opcode);
				activeFragment.append(frame.payload);
//				lastFragmentTime = System.currentTimeMillis();
			}
		}
	}

	/**
	 * 处理继续帧
	 */
	private void handleContinuationFrame(WebSocketFrame frame) throws IOException {
		synchronized (frameLock) {
			if (activeFragment == null) {
				if (Logger.isDebug()) Logger.debug("ws client: handle continue frame, ignore unexpected continuation frame");
				return;
			}

			activeFragment.append(frame.payload);
//			lastFragmentTime = System.currentTimeMillis();

			if (frame.fin) {
				// 分片消息完成
				processCompleteMessage(activeFragment.getType(), activeFragment.getData());
				activeFragment.clear();
				activeFragment = null;
			}
		}
	}

	private void processCompleteMessage(byte type, byte[] data) throws IOException {
		try {
			if (type == OPCODE_TEXT) {
				String message = new String(data, "UTF-8");
				if (listener != null) {
					listener.onMessage(message);
				}
				if (Logger.isDebug()) Logger.debug("ws client: recv complete text message: length=" + message.length());
			} else if (type == OPCODE_BINARY) {
				if (listener != null) {
					listener.onMessage(data);
				}
				if (Logger.isDebug()) Logger.debug("ws client: recv complete binary message: length=" + data.length);
			}
		} catch (UnsupportedEncodingException e) {
			throw new IOException("UTF-8 encoding not supported");
		}
	}

	private void processBufferedData() {
		try {
			while (true) {
				WebSocketFrame frame = tryReadFrame();
				if (frame != null) {
					handleFrame(frame);
				} else {
					// 没有完整帧可读，但可能有未完成的帧在处理中
					if (incompleteFrame != null) {
						if (Logger.isDebug()) Logger.debug("ws client: waiting for more data for incomplete frame: " + receivedPayloadLength + "/" + expectedPayloadLength);
					}
					break;
				}
			}
		} catch (IOException e) {
			// 重置未完成帧状态
			incompleteFrame = null;
			expectedPayloadLength = 0;
			receivedPayloadLength = 0;
			framePayloadBuffer = null;

			notifyError(e);
		}
	}

	/**
	 * 处理关闭帧
	 */
	private void handleCloseFrame(WebSocketFrame frame) throws IOException {
		int closeCode = 1000;
		String closeReason = "";

		if (frame.payload.length >= 2) {
			closeCode = ((frame.payload[0] & 0xFF) << 8) | (frame.payload[1] & 0xFF);
			if (frame.payload.length > 2) {
				closeReason = new String(frame.payload, 2, frame.payload.length - 2, "UTF-8");
			}
		}

		state = STATE_CLOSING;
		if (listener != null) {
			listener.onClose(closeCode, closeReason);
		}
		close();
	}

	/**
	 * 写入循环
	 */
	private void writeLoop() {
		try {
			while (running && state == STATE_OPEN && !interrupted) {
				OutgoingFrame frame = null;

				synchronized (sendLock) {
					while (sendQueue.isEmpty() && running && !interrupted) {
						try {
							sendLock.wait(1000); // 1秒超时
						} catch (InterruptedException e) {
							return;
						}
					}

					if (!sendQueue.isEmpty()) {
						frame = (OutgoingFrame) sendQueue.elementAt(0);
						sendQueue.removeElementAt(0);
					}
				}

				if (frame != null) {
					try {
						sendFrameInternal(frame);
					} catch (IOException e) {
						if (running) {
							notifyError(e);
						}
						break;
					}
				}
			}
		} finally {
			// 清空发送队列
			synchronized (sendLock) {
				sendQueue.removeAllElements();
			}
//			if (listener != null) {
//				listener.onClose(1000, "");
//			}
			close();
		}
	}

	/**
	 * 发送帧的内部实现
	 */
	private void sendFrameInternal(OutgoingFrame frame) throws IOException {
		ByteArrayOutputStream frameData = new ByteArrayOutputStream();

		// 第一个字节：FIN + RSV + OPCODE
		int firstByte = frame.opcode & 0x0F;
		if (frame.fin) {
			firstByte |= 0x80;
		}
		frameData.write(firstByte);

		// 负载长度和掩码
		int payloadLength = frame.payload.length;
		if (payloadLength < 126) {
			frameData.write(0x80 | payloadLength); // MASK=1
		} else if (payloadLength < 65536) {
			frameData.write(0x80 | 126); // MASK=1, extended length
			frameData.write((payloadLength >> 8) & 0xFF);
			frameData.write(payloadLength & 0xFF);
		} else {
			frameData.write(0x80 | 127); // MASK=1, extended length
			// 64位长度（简化为32位）
			frameData.write(0);
			frameData.write(0);
			frameData.write(0);
			frameData.write(0);
			frameData.write((payloadLength >> 24) & 0xFF);
			frameData.write((payloadLength >> 16) & 0xFF);
			frameData.write((payloadLength >> 8) & 0xFF);
			frameData.write(payloadLength & 0xFF);
		}

		// 掩码键
		byte[] maskKey = new byte[4];
		fillRandomBytes(maskKey);
		frameData.write(maskKey);

		// 掩码化的负载数据
		for (int i = 0; i < frame.payload.length; i++) {
			frameData.write(frame.payload[i] ^ maskKey[i % 4]);
		}

		synchronized (outputStream) {
			byte[] data = frameData.toByteArray();
			if (Logger.isDebug()) Logger.debug("ws client: sendFrameInternal bytes "+data.length);
			outputStream.write(data);
			outputStream.flush();
		}
	}

	/**
	 * 通知错误
	 */
	private void notifyError(Exception e) {
		if (Logger.isDebug()) Logger.debug("ws client: err exception=" + e);
		if (Logger.isInfo()) e.printStackTrace(); // 输出堆栈跟踪

		if (listener != null) {
			listener.onError(e);
		}
	}

	/**
	 * 关闭WebSocket连接
	 * 
	 * @return
	 */
	public synchronized boolean close() {
		close(1000, "");
		return true;
	}

	/**
	 * 关闭WebSocket连接
	 * 
	 * @param code   关闭代码
	 * @param reason 关闭原因
	 */
	public synchronized void close(int code, String reason) {
		if (state == STATE_CLOSED || state == STATE_CLOSING) {
			return;
		}

		running = false;
		interrupted = true;

		if (state == STATE_OPEN) {
			try {
				// 发送关闭帧
				ByteArrayOutputStream closePayload = new ByteArrayOutputStream();
				closePayload.write((code >> 8) & 0xFF);
				closePayload.write(code & 0xFF);
				if (reason != null && reason.length() > 0) {
					closePayload.write(reason.getBytes("UTF-8"));
				}
				queueFrame(new OutgoingFrame(OPCODE_CLOSE, closePayload.toByteArray(), true));

				// 等待一段时间让关闭帧发送完成
				try {
					Thread.sleep(100);
				} catch (InterruptedException e) {
					// 忽略
				}
			} catch (IOException e) {
				// 忽略发送关闭帧时的错误
			}
		}

		state = STATE_CLOSED;
		cleanup();
	}

	/**
	 * 清理资源
	 */
	private void cleanup() {
		if (Logger.isDebug()) Logger.debug("ws client: clean up!");

		// 清理未完成帧状态
		incompleteFrame = null;
		expectedPayloadLength = 0;
		receivedPayloadLength = 0;
		framePayloadBuffer = null;

		// 设置中断标志
		interrupted = true;

		// 通知所有等待的线程
		synchronized (sendLock) {
			sendLock.notifyAll();
		}

		try {
			if (inputStream != null) {
				inputStream.close();
			}
		} catch (IOException e) {
			// 忽略
		}

		try {
			if (outputStream != null) {
				outputStream.close();
			}
		} catch (IOException e) {
			// 忽略
		}

		try {
			if (socket != null) {
				socket.close();
			}
		} catch (IOException e) {
			// 忽略
		}
		if (Logger.isDebug()) Logger.debug("ws client: start waiting read/write thread stop");
		// 等待线程结束
		if (readerThread != null && readerThread.isAlive()) {
			try {
				long startTime = System.currentTimeMillis();
				while (readerThread.isAlive() && (System.currentTimeMillis() - startTime < 1000)) {
					readerThread.interrupt();
					Thread.sleep(50);
				}
			} catch (InterruptedException e) {
				// 忽略
			}
		}
		if (Logger.isDebug()) Logger.debug("ws client: read thread stopped");
		if (writerThread != null && writerThread.isAlive()) {
			try {
				long startTime = System.currentTimeMillis();
				while (writerThread.isAlive() && (System.currentTimeMillis() - startTime < 1000)) {
					writerThread.interrupt();
					Thread.sleep(50);
				}
			} catch (InterruptedException e) {
				// 忽略
			}
		}
		if (Logger.isDebug()) Logger.debug("ws client: write thread stopped");
	}

	/**
	 * 获取连接状态
	 */
	public int getState() {
		return state;
	}

	/**
	 * 检查连接是否打开
	 */
	public boolean isOpen() {
		return state == STATE_OPEN;
	}

	/**
	 * 设置是否使用轮询模式
	 */
	public void setPollingMode(boolean usePolling) {
		this.usePolling = usePolling;
	}

	/**
	 * 辅助方法：将字节转换为十六进制字符串（替代String.format）
	 */
	private String byteToHex(byte b) {
		int value = b & 0xFF;
		String hex = Integer.toHexString(value).toUpperCase();
		return hex.length() == 1 ? "0" + hex : hex;
	}

	/**
	 * 添加调试方法
	 */
	private void debugBuffer() {
		System.out.println("Buffer status: available=" + readBuffer.available() + ", readPos=" + readBuffer.getReadPosition());

		// 显示前几个字节的内容
		if (readBuffer.available() > 0) {
			System.out.print("Buffer content (hex): ");
			for (int i = 0; i < Math.min(16, readBuffer.available()); i++) {
				int b = readBuffer.peek(i);
				if (b != -1) {
					System.out.print(byteToHex((byte) b) + " ");
				}
			}
			System.out.println();
		}
	}
}