package com.joshvm.ws;

import com.joshvm.utils.Logger;
import com.joshvm.utils.Utils;

import java.io.IOException;

public class CircularBuffer {
	private byte[] buffer;
    private int readPos = 0;
    private int writePos = 0;
    private int count = 0;
    
    CircularBuffer(int size) {
        buffer = new byte[size];
    }
    
    public synchronized void write(byte[] data, int offset, int length) throws IOException {
        // 如果缓冲区不够，扩展它
        if (length > buffer.length - count) {
            expandBuffer(count + length);
        }
        
        for (int i = 0; i < length; i++) {
            buffer[writePos] = data[offset + i];
            writePos = (writePos + 1) % buffer.length;
            count++;
        }
    }
    
    private void expandBuffer(int minSize) {
        int newSize = Math.max(minSize * 2, buffer.length * 2);
        byte[] newBuffer = new byte[newSize];
        
        // 复制现有数据到新缓冲区
        for (int i = 0; i < count; i++) {
            newBuffer[i] = buffer[(readPos + i) % buffer.length];
        }
        
        buffer = newBuffer;
        readPos = 0;
        writePos = count;
        
        if (Logger.isTrace()) Logger.trace("CircularBuffer expand size to " + newSize);
    }
    
    public synchronized int read() {
        if (count == 0) {
            return -1;
        }
        
        int result = buffer[readPos] & 0xFF;
        readPos = (readPos + 1) % buffer.length;
        count--;
        return result;
    }
    
    public synchronized int available() {
        return count;
    }
    
    public synchronized int getReadPosition() {
        return readPos;
    }
    
    public synchronized void setReadPosition(int pos) {
        // 简化的位置重置
        if (pos >= 0 && pos < buffer.length) {
            // 计算需要跳过的字节数
            int skipBytes = 0;
            int currentPos = readPos;
            
            while (currentPos != pos && skipBytes < count) {
                currentPos = (currentPos + 1) % buffer.length;
                skipBytes++;
            }
            
            if (skipBytes <= count) {
                readPos = pos;
                count -= skipBytes;
            } else {
                // 重置到写入位置
                readPos = writePos;
                count = 0;
            }
        }
    }
    
    public synchronized int peek(int offset) {
        if (offset >= count || offset < 0) {
            return -1;
        }
        
        int pos = (readPos + offset) % buffer.length;
        return buffer[pos] & 0xFF;
    }
}
