package com.joshvm.utils;

import java.util.Calendar;
import java.util.Date;
import java.util.Hashtable;
import java.util.TimeZone;

/**
 * 日志工具类
 * 有限考虑性能, 建议用法如下,防止不必要的string拼接
 * if (Logger.isDebug()) Logger.debug
 * if (Logger.isInfo()) Logger.info
 * if (Logger.isWarn()) Logger.warn
 * e.printStack 也可用 if (Logger.isDebug()) 来判断处理
 *
 * currentLevel 可改成final以提高性能
 *
 * add getTimeStr(null)+ 到System.out.println(getTimeStr(null)+message)来每行log显示时间
 */
public class Logger {
    
    // 日志级别
    public static final int LEVEL_TRACE = 0;
    public static final int LEVEL_DEBUG = 1;
    public static final int LEVEL_INFO = 2;
    public static final int LEVEL_WARN = 3;
    public static final int LEVEL_ERROR = 4;
    public static final int LEVEL_FATAL = 5;
    
    // 当前日志级别
    private static int currentLevel = LEVEL_INFO;
    
    // 日志前缀
    private static final String[] LEVEL_NAMES = {"TRACE", "DEBUG", "INFO", "WARN", "ERROR", "FATAL"};

    private static Hashtable loggers = new Hashtable();
    private static Object lock = new Object();

    private String module;

    public static Logger getLogger(String module) {
        synchronized (lock) {
            Logger log = (Logger)loggers.get(module);
            if (log == null)
                return new Logger(null);
            return log;
        }
    }

    public Logger(String module) {
        this.module = module;
    }
    /**
     * 设置日志级别
     */
    public static void setLevel(int level) {
        if (level >= LEVEL_TRACE && level <= LEVEL_FATAL) {
            currentLevel = level;
        }
    }
    
    public static int getLevel() {
        return currentLevel;
    }

    // if (isTrace()) trace("msg") 避免拼接string带来的开销

    public static boolean isTrace() {
        return LEVEL_TRACE >= currentLevel;
    }
    // 减少开销
    public static void trace(String message) {
        if (LEVEL_TRACE >= currentLevel)
            System.out.println(message);
    }

    public static boolean isDebug() {
        return LEVEL_DEBUG >= currentLevel;
    }

    public static void debug(String message) {
        if (LEVEL_DEBUG >= currentLevel)
            System.out.println(message);
    }

    public static boolean isInfo() {
        return LEVEL_INFO >= currentLevel;
    }

    public static void info(String message) {
        if (LEVEL_INFO >= currentLevel)
            System.out.println(message);
    }

    public static boolean isWarn() {
        return LEVEL_WARN >= currentLevel;
    }

    public static void warn(String message) {
        if (LEVEL_WARN >= currentLevel)
            System.out.println(message);
    }

    public static boolean isError() {
        return LEVEL_ERROR >= currentLevel;
    }

    public static void error(String message) {
        if (LEVEL_ERROR >= currentLevel)
            System.out.println(message);
    }

    public static boolean isFatal() {
        return LEVEL_FATAL >= currentLevel;
    }

    public static void fatal(String message) {
        if (LEVEL_FATAL >= currentLevel)
            System.out.println(message);
    }

    public static void error(String message, Throwable throwable) {
        if (LEVEL_ERROR >= currentLevel)
            logTimestamp(LEVEL_ERROR, message + ": " + throwable.getMessage());
    }
    
    private static void logTimestamp(int level, String message) {
        if (level >= currentLevel) {
            long time = System.currentTimeMillis();
            long seconds = time / 1000;
            long milliseconds = time % 1000;
            String logMessage = seconds + "." + milliseconds + ":" + message;
            
            if (level >= LEVEL_ERROR) {
                System.err.println(logMessage);
            } else {
                System.out.println(logMessage);
            }
        }
    }

    /**
     * 获取当前时间戳
     */
    private static String getCurrentTimestamp() {
        long time = System.currentTimeMillis();
        long seconds = time / 1000;
        long milliseconds = time % 1000;
        
        // 简单的时间格式化
        return seconds + "." + milliseconds;
    }

    // 返回 2025-10-13 20:33:43.123 格式的时间字符串 yyyy-mm-dd hh:mm:ss.sss
    public static String getTimeStr(String zone) {
        Calendar cld = Calendar.getInstance();
        cld.setTime(new Date());
        cld.setTimeZone(TimeZone.getDefault());
        int y = cld.get(Calendar.YEAR);
        int m = cld.get(Calendar.MONTH)+1;
        int d = cld.get(Calendar.DAY_OF_MONTH);
        int h = cld.get(Calendar.HOUR_OF_DAY);
        int min = cld.get(Calendar.MINUTE);
        int s = cld.get(Calendar.SECOND);
        int sss = cld.get(Calendar.MILLISECOND);

        StringBuffer sb = new StringBuffer(25);// yyyy-mm-dd hh:mm:ss.sss
        if (y >= 0 && y < 1000) {
            sb.append('0');
            if (y < 100) sb.append('0');
            if (y < 10)  sb.append('0');
        }
        sb.append(y).append("-");
        if (m < 10) sb.append('0');    sb.append(m).append("-");
        if (d < 10) sb.append('0');    sb.append(d).append(" ");
        if (h < 10) sb.append('0');    sb.append(h).append(":");
        if (min < 10) sb.append('0');    sb.append(min).append(":");
        if (s < 10) sb.append('0');    sb.append(s).append(".");
        if (sss >= 0 && sss < 100) {
            sb.append('0');
            if (sss < 10) sb.append('0');
        }
        sb.append(sss).append(" ");

        return sb.toString();
//        TimeZone tz = TimeZone.getTimeZone(zone); // GMT as default
//        Calendar cld = Calendar.getInstance();
//        cld.setTime(new Date(System.currentTimeMillis()));
//        cld.setTimeZone(tz);
//
//        int year = cld.get(Calendar.YEAR);
    }

    /**
     * 记录方法调用日志
     */
    public static void logMethodCall(String className, String methodName, String parameters) {
        debug("Calling method: " + className + "." + methodName + "(" + parameters + ")");
    }
    
    /**
     * 记录方法返回日志
     */
    public static void logMethodReturn(String className, String methodName, String returnValue) {
        debug("Method return: " + className + "." + methodName + " = " + returnValue);
    }
    
    /**
     * 记录性能日志
     */
    public static void logPerformance(String operation, long startTime, long endTime) {
        long duration = endTime - startTime;
        info("Performance statistics: " + operation + " spend " + duration + "ms");
    }
}
